<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Teacher;
use App\Models\Enrollment;
use App\Models\Section;
use Carbon\Carbon;
use Illuminate\Support\Facades\Mail;
use App\Mail\EnrollmentChangeNotification;

class CheckEnrollmentChanges extends Command
{
    protected $signature = 'enrollments:check-changes';
    protected $description = 'Check for enrollment changes and notify teachers';

    public function handle()
    {
        $fourHoursAgo = Carbon::now()->subHours(4);

        Teacher::chunk(100, function ($teachers) use ($fourHoursAgo) {
            foreach ($teachers as $teacher) {
                $newEnrollments = [];
                $droppedEnrollments = [];

                foreach ($teacher->sections as $section) {
                    // New enrollments (created or updated to 'A' status in the last 4 hours)
                    $newEnrollments = array_merge($newEnrollments, $section->enrollments()
                        ->with('student', 'section.course') // Eager load the relationships
                        ->where(function ($query) use ($fourHoursAgo) {
                            $query->where('created_at', '>=', $fourHoursAgo);
                        })
                        ->where('status', 'A')
                        ->get()
                        ->map(function ($enrollment) {
                            $studentName = $enrollment->student
                                ? trim($enrollment->student->first_name . ' ' . $enrollment->student->last_name)
                                : 'Unknown Student';
                            $studentEmail = $enrollment->student->email ?? 'Unknown Email';
                            return [
                                'student_info' => "{$studentName} ({$studentEmail})",
                                'course_name' => $enrollment->section->course->name ?? 'Unknown Course'
                            ];
                        })
                        ->toArray());

                    // Dropped enrollments (updated to 'R' status in the last 4 hours)
                    $droppedEnrollments = array_merge($droppedEnrollments, $section->enrollments()
                        ->with('student', 'section.course') // Eager load the relationships
                        ->where('updated_at', '>=', $fourHoursAgo)
                        ->where('status', 'R')
                        ->get()
                        ->map(function ($enrollment) {
                            $studentName = $enrollment->student
                                ? trim($enrollment->student->first_name . ' ' . $enrollment->student->last_name)
                                : 'Unknown Student';
                            $studentEmail = $enrollment->student->email ?? 'Unknown Email';
                            return [
                                'student_info' => "{$studentName} ({$studentEmail})",
                                'course_name' => $enrollment->section->course->name ?? 'Unknown Course'
                            ];
                        })
                        ->toArray());
                }

                if (count($newEnrollments) > 0 || count($droppedEnrollments) > 0) {
                    Mail::to($teacher->email)->send(new EnrollmentChangeNotification($newEnrollments, $droppedEnrollments));
                    $this->info("Notification sent to teacher: {$teacher->name}");
                }
            }
        });

        $this->info('Enrollment check completed.');
    }
}
