<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Providers\RouteServiceProvider;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;
use Illuminate\View\View;
use App\Models\School;
use App\Models\Teacher;
use App\Models\Provider;
use App\Mail\RegistrationWelcomeEmail;
use Illuminate\Support\Facades\Mail;
use Illuminate\Validation\Rule;

class RegisteredUserController extends Controller
{
    /**
     * Display the registration view.
     */
    public function create()
    {
        if (auth()->user()->role_id != 1) {
            return redirect(RouteServiceProvider::HOME)->with('error','You do not have permission to register users')->with('title', 'Dashboard');
        }
        return view('auth.register')
        ->with('schools', School::orderby('name')->get())
        ->with('providers', Provider::orderby('name')->get())
        ->with('teachers', Teacher::orderby('last_name')->get())
        ->with('title', 'Register Users');
    }

    /**
     * Handle an incoming registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            // Basic user information validation
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:'.User::class],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],

            // Validate role_id to ensure it's one of the allowed values
            'role_id' => ['required', 'integer', Rule::in([2, 3, 4])],

            // Conditional validation for provider_id
            'provider_id' => [
                // Make provider_id required only if role_id is 2 (Provider)
                Rule::requiredIf(function () use ($request) {
                    return $request->role_id == 2;
                }),
                // When role_id is 2, provider_id must be an integer greater than 0
                Rule::when($request->role_id == 2, ['integer', 'min:1']),
            ],

            // Conditional validation for school_id
            'school_id' => [
                // Make school_id required only if role_id is 3 (School Personnel)
                Rule::requiredIf(function () use ($request) {
                    return $request->role_id == 3;
                }),
                // When role_id is 3, school_id must be an integer greater than 0
                Rule::when($request->role_id == 3, ['integer', 'min:1']),
            ],

            // Conditional validation for teacher_id
            'teacher_id' => [
                // Make teacher_id required only if role_id is 4 (Teacher)
                Rule::requiredIf(function () use ($request) {
                    return $request->role_id == 4;
                }),
                // When role_id is 4, teacher_id must be an integer greater than 0
                Rule::when($request->role_id == 4, ['integer', 'min:1']),
            ],
        ], [
            // Custom error messages for clarity
            'provider_id.required' => 'The provider field is required when the role is Provider.',
            'provider_id.min' => 'Please select a valid provider.',
            'school_id.required' => 'The school field is required when the role is School Personnel.',
            'school_id.min' => 'Please select a valid school.',
            'teacher_id.required' => 'The teacher field is required when the role is Teacher.',
            'teacher_id.min' => 'Please select a valid teacher.',
        ]);

        // Create the user with validated data
        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'school_id' => $request->school_id,
            'provider_id' => $request->provider_id,
            'teacher_id' => $request->teacher_id,
            'role_id' => $request->role_id,
        ]);

        // Send welcome email to the new user
        Mail::to($user->email)->send(new RegistrationWelcomeEmail($request->email, $request->password));

        // Trigger the Registered event
        event(new Registered($user));

        // Redirect to the home page with a success message
        return redirect(RouteServiceProvider::HOME)->with('success', 'User registered successfully!')->with('title', 'Dashboard');
    }
}
