<?php

namespace App\Http\Controllers;

use App\Models\Dashboard;
use App\Models\Department;
use App\Models\Enrollment;
use App\Models\Year;
use App\Models\Student;
use App\Models\Section;
use App\Traits\FetchesStudentsWithoutSSID;

use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    use FetchesStudentsWithoutSSID;

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $yearId = Year::latest('id')->first()->id;
        $schoolId = 0;
        $providerId = 0;
        $teacherId = 0;

        $roleId = auth()->user()->role_id;

        switch ($roleId) {
            case 1: // Admin
                break;
            case 2: // Provider
                $providerId = auth()->user()->provider_id;
                break;
            case 3: // School
                $schoolId = auth()->user()->school_id;
                break;
            default: // Teacher
                $teacherId = auth()->user()->teacher_id;
                break;
        }
        $departments = Department::all();
        foreach($departments as $d) {
            $d->totalEnrollments = $this->getEnrollments($yearId, $schoolId, $providerId, $d->id, 0, $teacherId, 'A', 'A', 0)->count();
        }

        $departments = $departments->where('totalEnrollments', '>', 0)->values();

        $enrollments = $this->getEnrollments($yearId, $schoolId, $providerId, 0, 0, $teacherId, 'A', 'A', 0);

        $cancelledSectionEnrollments = $enrollments->where("is_cancelled", true);
        $powerSchoolEnrollmentsNeedingUpdate = $enrollments->where("ps_updated", false);
        $waitListEnrollmentsReadyToAdd = $this->getEnrollments($yearId, $schoolId, $providerId, 0, 0, $teacherId, 'W', 'A', 0)->sortBy('id');
        $totalWelcomeLettersNeedtoSend = Enrollment::where('sent_welcome_letter', false)
            ->whereHas('section', function($query) {
                $query->whereIn('term', ['F', 'Y']);
            })
            ->when($schoolId, function ($query) use ($schoolId) {
                return $query->whereHas('student', function ($q) use ($schoolId) {
                    $q->where('school_id', $schoolId);
                });
            })
            ->when($providerId, function ($query) use ($providerId) {
                return $query->whereHas('section.course', function ($q) use ($providerId) {
                    $q->where('provider_id', $providerId);
                });
            })
            ->when($teacherId, function ($query) use ($teacherId) {
                return $query->whereHas('section', function ($q) use ($teacherId) {
                    $q->where('teacher_id', $teacherId);
                });
            })
            ->where('year_id', $yearId)
            ->where('status', 'A')
            ->count();

        $studentsWithoutSSID = $this->getStudentsWithoutSSID($schoolId);


        $data = [
            'title' => 'Dashboard',
            'departments' => $departments,
            'cancelledSectionEnrollments' => $cancelledSectionEnrollments->take(4),
            'totalCancelledSectionEnrollments' => $cancelledSectionEnrollments->count(),
            'powerSchoolEnrollmentsNeedingUpdate' => $powerSchoolEnrollmentsNeedingUpdate,
            'waitListEnrollmentsReadyToAdd' => $waitListEnrollmentsReadyToAdd,
            'totalWelcomeLettersNeedtoSend' => $totalWelcomeLettersNeedtoSend,
            'studentsWithoutSSID' => $studentsWithoutSSID,
        ];

        if ($roleId == 1) {
            $view = 'admin.dashboard.admin';
        } elseif ($roleId == 2) {
            $view = 'admin.dashboard.provider';
        } elseif ($roleId == 3) {
            $view = 'admin.dashboard.school';
        } else {
            $view = 'admin.dashboard.teacher';
        }
        return view($view, $data);
    }
    /**
     * Get enrollments based on various criteria.
     */
    private function getEnrollments($yearId, $schoolId, $providerId, $departmentId, $sectionId, $teacherId, $status, $term, $grade)
        {
            return Enrollment::where('year_id', $yearId)
                ->when($schoolId, function ($query) use ($schoolId) {
                    return $query->whereHas('student', function ($q) use ($schoolId) {
                        $q->where('school_id', $schoolId);
                    });
                })
                ->when($providerId, function ($query) use ($providerId) {
                    return $query->whereHas('section.course', function ($q) use ($providerId) {
                        $q->where('provider_id', $providerId);
                    });
                })
                ->when($departmentId, function ($query) use ($departmentId) {
                    return $query->whereHas('section.course', function ($q) use ($departmentId) {
                        $q->where('department_id', $departmentId);
                    });
                })
                ->when($sectionId, function ($query) use ($sectionId) {
                    return $query->where('section_id', $sectionId);
                })
                ->when($teacherId, function ($query) use ($teacherId) {
                    return $query->whereHas('section', function ($q) use ($teacherId) {
                        $q->where('teacher_id', $teacherId);
                    });
                })
                ->when($status !== 'ALL', function ($query) use ($status) {
                    return $query->where('status', $status);
                })
                ->when($term !== 'A', function ($query) use ($term) {
                    return $query->where('term', $term);
                })
                ->when($grade, function ($query) use ($grade) {
                    return $query->where('grade', $grade);
                })
                ->get();
        }
}
