<?php

namespace App\Http\Controllers;

use App\Models\Student;
use App\Models\School;
use App\Models\Contact;
use App\Models\Section;
use App\Models\Enrollment;
use App\Http\Requests\StoreStudentRequest;
use App\Http\Requests\UpdateStudentRequest;
use App\Traits\FetchesStudentsWithoutSSID;
use Illuminate\Support\Facades\Auth;

class StudentController extends Controller
{
    use FetchesStudentsWithoutSSID;

    public function index()
    {
        $query = Student::query();

        if (!in_array(auth()->user()->role_id, [1, 2])) {
            $query->where('school_id', auth()->user()->school_id);
        }

        $students = $query->orderBy('last_name')->get();

        return view('admin.students.index', compact('students'))
            ->with('title', 'Students');
    }

    public function create()
    {
        $schools = $this->getSchools();
        return view('admin.students.form', compact('schools'))
            ->with('action', 'create')
            ->with('title', 'Create Student');
    }

    public function store(StoreStudentRequest $request)
    {
        Student::create($request->validated());
        return redirect(route('students.index'))->with('success', 'Student created successfully!');
    }

    public function show(Student $student)
    {
        $contacts = Contact::where('school_id', $student->school_id)
            ->orderBy('last_name')
            ->get();

        return view('admin.students.show', compact('student', 'contacts'))
            ->with('title', "View {$student->first_name} {$student->last_name}");
    }

    public function edit(Student $student)
    {
        $schools = $this->getSchools();
        return view('admin.students.form', compact('student', 'schools'))
            ->with('action', 'edit')
            ->with('title', "Edit {$student->first_name} {$student->last_name}");
    }

    public function update(UpdateStudentRequest $request, Student $student)
    {
        $student->update($request->validated());
        return redirect(route('students.index'))
            ->with('success', "Student {$student->first_name} {$student->last_name} updated successfully!");
    }

    public function delete(Student $student)
    {
        if ($student->enrollments()->exists()) {
            return redirect(route('students.index'))
                ->with('error', 'Cannot delete student that has enrollments.');
        }

        return view('admin.students.delete', compact('student'))
            ->with('title', "Delete {$student->first_name} {$student->last_name}");
    }

    public function destroy(Student $student)
    {
        $student->delete();
        return redirect(route('students.index'))
            ->with('success', "Student {$student->first_name} {$student->last_name} deleted successfully!");
    }

    public function bulkUpload()
    {
        return view('admin.students.bulkupload')
            ->with('action', 'create')
            ->with('title', 'Bulk Upload Students');
    }

    public function needsSSID()
    {
        $user = Auth::user();
        $schoolId = $user->role_id == 3 ? $user->school_id : null;

        $students = $this->getStudentsWithoutSSID($schoolId);

        if ($user->role_id == 2) {
            $sectionIds = Section::where('provider_id', $user->provider_id)->pluck('id');
            $studentIds = Enrollment::whereIn('section_id', $sectionIds)->pluck('student_id');
            $students = $students->whereIn('id', $studentIds);
        }

        return view('admin.students.needs_ssid', compact('students'))
            ->with('title', 'Students Needing SSID (' . $students->count() . ')');
    }

    private function getSchools()
    {
        $query = School::query();

        if (auth()->user()->role_id == 3) {
            $query->where('id', auth()->user()->school_id);
        }

        return $query->orderBy('name')->get();
    }
}
