<?php

namespace App\Http\Controllers;

use App\Models\WelcomeLetter;
use App\Models\Section;
use App\Models\Student;
use App\Models\Teacher;
use App\Models\Enrollment;
use App\Models\Year;
use App\Models\School;
use App\Models\Contact;
use App\Http\Requests\StoreWelcomeLetterRequest;
use App\Http\Requests\UpdateWelcomeLetterRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Collection;
use File;
use App\Services\EmailService;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;

class WelcomeLetterController extends Controller
{
    protected $emailService;

    public function __construct(EmailService $emailService)
    {
        $this->emailService = $emailService;
    }

    public function index()
    {
        $user = Auth::user();
        $welcomeLettersQuery = WelcomeLetter::with('section.provider', 'section.course', 'section.teacher', 'section.year')
            ->whereHas('section');

        switch ($user->role_id) {
            case 1: // Super Admin
                $welcome_letters = $welcomeLettersQuery->get();
                break;
            case 2: // Provider Admin
                $welcome_letters = $welcomeLettersQuery
                    ->whereHas('section', function ($query) use ($user) {
                        $query->where('provider_id', $user->provider_id);
                    })
                    ->get();
                break;
            case 3: // School Admin
                return redirect()->route('dashboard')->with('error', 'You do not have permission to view welcome letters.');
            case 4: // Teacher
                $welcome_letters = $welcomeLettersQuery
                    ->whereHas('section', function ($query) use ($user) {
                        $query->where('teacher_id', $user->teacher_id);
                    })
                    ->get();
                break;
            default:
                return redirect()->route('dashboard')->with('error', 'Invalid user role.');
        }

        $pendingCounts = Section::getPendingWelcomeLettersCount();

        return view('admin.welcome-letters.index', compact('welcome_letters', 'pendingCounts'))
            ->with('title', 'Welcome Letters');
    }

    public function show(WelcomeLetter $welcomeLetter)
    {
        if($this->checkPermissions('show', $welcomeLetter->section->teacher_id) == false){
            return redirect(route('dashboard'))->with('error','You do not have permission to view that welcome letter.');
        }

        $facilitators = $this->getFacilitatorsForSection($welcomeLetter->section_id);

        return view('admin.welcome-letters.show', [
            'welcomeLetter' => $welcomeLetter,
            'facilitators' => $facilitators
        ])
        ->with('title', 'View Welcome Letter: ' . $welcomeLetter->section->course->name . " " . $welcomeLetter->section->year->year . " " . $welcomeLetter->section->termDesc());
    }

    public function create()
    {
        // Check permissions for creating a welcome letter
        if ($this->checkPermissions('create', null) == false) {
            return redirect(route('dashboard'))->with('error', 'You do not have permission to create welcome letters.');
        }

        $user = Auth::user();

        // Query to filter sections that do not have a welcome letter and year_id > 1
        $sectionsQuery = Section::with(['course', 'year', 'teacher'])
            ->whereDoesntHave('welcomeLetter')
            ->where('year_id', '>', 1);

        // Filter sections based on the user's role
        switch ($user->role_id) {
            case 1: // Super Admin
                $sections = $sectionsQuery->get();
                break;
            case 2: // Provider Admin
                $sections = $sectionsQuery
                    ->where('provider_id', $user->provider_id)
                    ->get();
                break;
            case 4: // Teacher
                $sections = $sectionsQuery
                    ->where('teacher_id', $user->teacher_id)
                    ->get();
                break;
            default:
                return redirect()->route('dashboard')->with('error', 'Invalid user role.');
        }

        // Return the create form view with the filtered sections
        return view('admin.welcome-letters.form', compact('sections'))
            ->with('action', 'create')
            ->with('title', 'Create Welcome Letter');
    }


    public function update(UpdateWelcomeLetterRequest $request, WelcomeLetter $welcomeLetter)
    {
        if($this->checkPermissions('update', $welcomeLetter->section->teacher_id) == false){
            return redirect(route('dashboard'))->with('error','You do not have permission to update that welcome letter.');
        }

        try {
            DB::beginTransaction();

            $welcomeLetterData = [
                'text' => $request->text,
                'updated_by' => auth()->id()
            ];

            // Handle all three files
            for ($i = 1; $i <= 3; $i++) {
                $fileKey = "file{$i}";
                $titleKey = "file{$i}_title";
                $oldFileKey = "file{$i}_old";

                // Check if file should be removed
                if ($welcomeLetter->$fileKey && !$request->hasFile($fileKey) && !$request->has($oldFileKey)) {
                    $oldFilePath = public_path('uploads/welcome-letter-files/' . $welcomeLetter->$fileKey);
                    if (File::exists($oldFilePath)) {
                        File::delete($oldFilePath);
                    }
                    $welcomeLetterData[$fileKey] = null;
                    $welcomeLetterData[$titleKey] = null;
                }

                // Check if new file is uploaded
                if ($request->hasFile($fileKey)) {
                    // Remove old file if exists
                    if ($welcomeLetter->$fileKey) {
                        $oldFilePath = public_path('uploads/welcome-letter-files/' . $welcomeLetter->$fileKey);
                        if (File::exists($oldFilePath)) {
                            File::delete($oldFilePath);
                        }
                    }

                    // Upload new file
                    $file = $request->file($fileKey);
                    $timestamp = now()->format('Y_m_d_His_u');
                    $random = substr(str_shuffle('abcdefghjklmnopqrstuvwxyz'), 0, 6);
                    $extension = $file->getClientOriginalExtension();
                    $filename = "wl_{$timestamp}_{$random}.{$extension}";

                    $uploadPath = public_path('uploads/welcome-letter-files');

                    // Create directory if it doesn't exist
                    if (!File::exists($uploadPath)) {
                        File::makeDirectory($uploadPath, 0755, true);
                    }

                    $file->move($uploadPath, $filename);

                    $welcomeLetterData[$fileKey] = $filename;
                    $welcomeLetterData[$titleKey] = $request->input($titleKey);
                }
            }

            $welcomeLetter->update($welcomeLetterData);

            DB::commit();

            return redirect()->route('welcome-letters.show', $welcomeLetter)
                ->with('success', 'Welcome letter updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to update welcome letter: ' . $e->getMessage());
            return back()->with('error', 'Failed to update welcome letter. Please try again.');
        }
    }

    public function store(StoreWelcomeLetterRequest $request)
    {
        if($this->checkPermissions('store', null) == false){
            return redirect(route('dashboard'))->with('error','You do not have permission to create welcome letters.');
        }

        $section = Section::findOrFail($request->section_id);

        if (auth()->user()->role_id == 4 && auth()->user()->teacher_id != $section->teacher_id) {
            return redirect()->route('dashboard')->with('error', 'You can only create welcome letters for your own sections.');
        }

        try {
            DB::beginTransaction();

            $welcomeLetterData = [
                'section_id' => $request->section_id,
                'text' => $request->text,
                'created_by' => auth()->id(),
                'updated_by' => auth()->id()
            ];

            // Handle all three files
            for ($i = 1; $i <= 3; $i++) {
                $fileKey = "file{$i}";
                $titleKey = "file{$i}_title";

                if ($request->hasFile($fileKey)) {
                    $file = $request->file($fileKey);
                    $timestamp = now()->format('Y_m_d_His_u');
                    $random = substr(str_shuffle('abcdefghjklmnopqrstuvwxyz'), 0, 6);
                    $extension = $file->getClientOriginalExtension();
                    $filename = "wl_{$timestamp}_{$random}.{$extension}";

                    $uploadPath = public_path('uploads/welcome-letter-files');

                    // Create directory if it doesn't exist
                    if (!File::exists($uploadPath)) {
                        File::makeDirectory($uploadPath, 0755, true);
                    }

                    $file->move($uploadPath, $filename);

                    $welcomeLetterData[$fileKey] = $filename;
                    $welcomeLetterData[$titleKey] = $request->input($titleKey);
                }
            }

            $welcomeLetter = WelcomeLetter::create($welcomeLetterData);

            DB::commit();

            return redirect()->route('welcome-letters.show', $welcomeLetter)
                ->with('success', 'Welcome letter created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to create welcome letter: ' . $e->getMessage());
            return back()->with('error', 'Failed to create welcome letter. Please try again.');
        }
    }

    public function edit(WelcomeLetter $welcomeLetter)
    {
        // Check permissions for editing a welcome letter
        if ($this->checkPermissions('edit', $welcomeLetter->section->teacher_id) == false) {
            return redirect(route('dashboard'))->with('error', 'You do not have permission to edit that welcome letter.');
        }

        $user = Auth::user();

        // Query to filter sections that do not have a welcome letter, have year_id > 1, and include the current section
        $sectionsQuery = Section::with(['course', 'year', 'teacher'])
            ->where(function ($query) use ($welcomeLetter) {
                $query->whereDoesntHave('welcomeLetter')
                      ->orWhere('id', $welcomeLetter->section_id); // Include the current section
            })
            ->where('year_id', '>', 1);

        // Filter sections based on the user's role
        switch ($user->role_id) {
            case 1: // Super Admin
                $sections = $sectionsQuery->get();
                break;
            case 2: // Provider Admin
                $sections = $sectionsQuery
                    ->where('provider_id', $user->provider_id)
                    ->get();
                break;
            case 4: // Teacher
                $sections = $sectionsQuery
                    ->where('teacher_id', $user->teacher_id)
                    ->get();
                break;
            default:
                return redirect()->route('dashboard')->with('error', 'Invalid user role.');
        }

        // Return the edit form view with the filtered sections and the welcome letter
        return view('admin.welcome-letters.form', compact('welcomeLetter', 'sections'))
            ->with('action', 'edit')
            ->with('title', 'Edit Welcome Letter: ' . $welcomeLetter->section->course->name . " " .
                $welcomeLetter->section->year->year . " " . $welcomeLetter->section->termDesc());
    }

    public function delete(WelcomeLetter $welcomeLetter)
    {
        // Check if the user has permission to delete this welcome letter
        $user = Auth::user();

        // Allow deletion only if the user is the creator or has role_id = 1 (Super Admin)
        if ($user->role_id != 1 && $user->id != $welcomeLetter->created_by) {
            return redirect()->route('welcome-letters.index')
                ->with('error', 'You do not have permission to delete this welcome letter.');
        }

        try {
            // Attempt to delete the welcome letter
            $welcomeLetter->delete();

            return redirect()->route('welcome-letters.index')
                ->with('success', 'Welcome letter deleted successfully.');
        } catch (\Exception $e) {
            // Log the error and return with failure
            Log::error('Failed to delete welcome letter: ' . $e->getMessage());

            return redirect()->route('welcome-letters.index')
                ->with('error', 'Failed to delete the welcome letter. Please try again.');
        }
    }


    public function destroy(WelcomeLetter $welcomeLetter)
    {
        if($this->checkPermissions('destroy', $welcomeLetter->section->teacher_id) == false){
            return redirect(route('dashboard'))->with('error','You do not have permission to delete that welcome letter.');
        }

        try {
            DB::beginTransaction();

            $welcomeLetter->delete();

            DB::commit();

            return redirect()->route('welcome-letters.index')
                ->with('success', 'Welcome letter deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to delete welcome letter: ' . $e->getMessage());
            return back()->with('error', 'Failed to delete welcome letter. Please try again.');
        }
    }

    public function bulkSend(WelcomeLetter $welcomeLetter)
    {
        if($this->checkPermissions('bulksend', $welcomeLetter->section->teacher_id) == false){
            return redirect(route('dashboard'))->with('error','You do not have permission to view that welcome letter.');
        }
        $enrollments = Section::find($welcomeLetter->section_id)->enrollments()->where('status', 'A')->get();
        foreach($enrollments as $e) {
            $e->first_name = $e->student->first_name;
            $e->last_name = $e->student->last_name;
        }

        $facilitators = $this->getFacilitatorsForSection($welcomeLetter->section_id);

        return view('admin.welcome-letters.bulk-send', [
            'welcomeLetter' => $welcomeLetter,
            'enrollments' => $enrollments->sortBy('first_name')->sortBy('last_name'),
            'facilitators' => $facilitators
        ])
        ->with('action', 'bulksend')
        ->with('title', 'Bulk Send Welcome Letter: ' . $welcomeLetter->section->course->name . " " . $welcomeLetter->section->year->year . " " . $welcomeLetter->section->termDesc());
    }


    public function bulkSendConfirm(Request $request, WelcomeLetter $welcomeLetter)
    {
        Log::info('Starting bulkSendConfirm', [
            'enrollment_ids' => $request->input('enrollments'),
            'facilitator_ids' => $request->input('facilitators')
        ]);
    
        $enrollmentIds = $request->input('enrollments');
        $facilitatorIds = $request->input('facilitators');
        $bccEmails = [];
    
        if (!empty($enrollmentIds)) {
            foreach ($enrollmentIds as $enrollmentId) {
                $studentEmail = Enrollment::find($enrollmentId)->student->email;
                $bccEmails[] = $studentEmail;
            }
    
            $section = Section::findOrFail($welcomeLetter->section_id);
            $teacherEmail = $section->teacher->email;
    
            $bccEmailString = implode(';', $bccEmails);
    
            $this->emailService->sendWelcomeEmail(
                $section->id,
                $teacherEmail,
                $teacherEmail,
                $section->teacher->first_name . " " . $section->teacher->last_name,
                'Welcome to ' . $section->course->name,
                $welcomeLetter->text,
                $bccEmailString
            );
    
            Enrollment::whereIn('id', $enrollmentIds)->update(['sent_welcome_letter' => true]);
        }
    
        Log::info('Before facilitator check', [
            'has_facilitator_ids' => !empty($facilitatorIds),
            'facilitator_ids' => $facilitatorIds
        ]);
    
        if (!empty($facilitatorIds)) {
            $section = Section::findOrFail($welcomeLetter->section_id);
            $teacherEmail = $section->teacher->email;
            
            Log::info('Processing facilitators', ['facilitator_ids' => $facilitatorIds]);
            
            $facilitators = Contact::whereIn('id', $facilitatorIds)
                ->where('is_facilitator', 1)
                ->get();
                
            Log::info('Found facilitators', ['count' => $facilitators->count()]);
    
            foreach ($facilitators as $facilitator) {
                $students = Student::where('school_id', $facilitator->school_id)
                    ->whereHas('enrollments', function ($query) use ($section) {
                        $query->where('section_id', $section->id);
                    })
                    ->orderBy('last_name')
                    ->orderBy('first_name')
                    ->get();
                    
                Log::info('Processing school', [
                    'school_id' => $facilitator->school_id,
                    'student_count' => $students->count(),
                    'student_ids' => $students->pluck('id')
                ]);
    
                if ($students->isNotEmpty()) {
                    $studentList = "As of now, students enrolled in this section from your school:\n\n";
                    foreach ($students as $student) {
                        $studentList .= "- {$student->last_name}, {$student->first_name}\n";
                    }
                    $studentList .= "\n";
    
                    $emailContent = $studentList . $welcomeLetter->text;
    
                    $this->emailService->sendWelcomeEmail(
                        $section->id,
                        $facilitator->email,
                        $teacherEmail,
                        $section->teacher->first_name . " " . $section->teacher->last_name,
                        'Facilitator Copy: Welcome to ' . $section->course->name,
                        $emailContent,
                        null
                    );
    
                    $updateCount = Enrollment::where('section_id', $section->id)
                        ->whereIn('student_id', $students->pluck('id'))
                        ->update(['sent_facilitator_copy' => 1]);
                        
                    Log::info('Updated enrollments', [
                        'school_id' => $facilitator->school_id,
                        'update_count' => $updateCount,
                        'section_id' => $section->id,
                        'student_ids' => $students->pluck('id')
                    ]);
                }
            }
        }
    
        return redirect(route('welcome-letters.index'))->with('success', 'Welcome Letter emails sent successfully!');
    }
    private function getFacilitatorsForSection($sectionId)
    {
       return Contact::where('is_facilitator', 1)
           ->join('schools', 'contacts.school_id', '=', 'schools.id')
           ->join('students', 'students.school_id', '=', 'schools.id')
           ->join('enrollments', 'enrollments.student_id', '=', 'students.id')
           ->where('enrollments.section_id', $sectionId)
           ->where('enrollments.status', 'A')
           ->select('contacts.*', 'schools.name as school_name',
               DB::raw('MAX(enrollments.sent_facilitator_copy) as sent_copy'))
           ->groupBy('contacts.id', 'contacts.first_name', 'contacts.last_name', 
                    'contacts.school_id', 'contacts.position_id', 'contacts.email',
                    'contacts.phone', 'contacts.ext', 'contacts.is_facilitator',
                    'contacts.created_at', 'contacts.updated_at',
                    'schools.name')
           ->orderBy('schools.name', 'asc')
           ->orderBy('contacts.last_name', 'asc')
           ->orderBy('contacts.first_name', 'asc')
           ->get();
    }

    private function checkPermissions($action, $welcomeLetterTeacherId)
    {
        switch ($action) {
            case 'show':
            case 'edit':
            case 'update':
            case 'delete':
            case 'destroy':
            case 'bulksend':
                if (auth()->user()->role_id == 3) {
                    return false;
                }
                if (auth()->user()->teacher_id != $welcomeLetterTeacherId && (auth()->user()->role_id != 1 && auth()->user()->role_id != 2)) {
                    return false;
                }
                break;
            default:
                if (auth()->user()->role_id == 3) {
                    return false;
                }
        }
        return true;
    }
}
