<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\DB;

class Section extends Model
{
    use HasFactory;

    protected $fillable = [
        'year_id',
        'course_id',
        'provider_id',
        'term',
        'teacher_id',
        'enrollment_max',
        'delivery',
        'minimum_grade',
        'dual_credit',
        'description',
        'is_moodle',
        'moodle_short_name',
        'is_cancelled',
        'restricted_school_ids'
    ];

    protected $casts = [
        'restricted_school_ids' => 'array'
    ];

    public function enrollments(): HasMany
    {
        return $this->hasMany(Enrollment::class, 'section_id')->where('status', 'A');
    }

    public function enrollmentsAnyStatus(): HasMany
    {
        return $this->hasMany(Enrollment::class, 'section_id');
    }

    public function openSeats()
    {
        return $this->enrollment_max - $this->enrollments->count();
    }

    public function openSeatsDesc()
    {
        $openSeats = $this->enrollment_max - $this->enrollments->count();
        if ($openSeats > 0) {
            if ($openSeats == 1) {
                return '1 seat left';
            } else {
                return $openSeats . ' seats left';
            }
        } else {
            return 'FULL - ' . $this->enrollments->count() - $this->enrollment_max . ' on waitlist';
        }
    }

    public function teacher(): BelongsTo
    {
        return $this->belongsTo(Teacher::class, 'teacher_id');
    }

    public function course(): BelongsTo
    {
        return $this->belongsTo(Course::class, 'course_id');
    }

    public function year(): BelongsTo
    {
        return $this->belongsTo(Year::class, 'year_id');
    }

    public function provider(): BelongsTo
    {
        return $this->belongsTo(Provider::class, 'provider_id');
    }

    public function termDesc()
    {
        if ($this->term === 'Y') {
            return 'Full Year';
        } elseif ($this->term === 'F') {
            return 'Fall';
        } else {
            return 'Spring';
        }
    }

    public function welcomeLetter()
    {
        return $this->hasOne(WelcomeLetter::class, 'section_id');
    }

    public static function getPendingWelcomeLettersCount($yearId = null)
    {
        $query = Enrollment::select('section_id', DB::raw('COUNT(*) as pending_count'))
            ->where('sent_welcome_letter', false)
            ->where('status', 'A')
            ->groupBy('section_id');

        if ($yearId) {
            $query->where('year_id', $yearId);
        }

        return $query->pluck('pending_count', 'section_id')->toArray();
    }

    public static function sectionSQLBuilder($yearId, $providerId, $departmentId, $teacherId, $term, $dualCredit, $availability, $schoolId = 0)
    {
        $sql = DB::table("sections as s")
            ->join("teachers as t", "t.id", "=", "s.teacher_id")
            ->join("courses as c", "c.id", "=", "s.course_id")
            ->join("departments as d", "d.id", "=", "c.department_id")
            ->join("providers as p", "p.id", "=", "s.provider_id")
            ->leftJoin("enrollments as e", "e.section_id", "=", "s.id")
            ->select(
                "s.*",
                "d.id as departmentid",
                "p.id as providerid",
                "t.id as teacherId",
                DB::raw("COUNT(CASE WHEN e.status = 'A' THEN e.id END) as enrollment_count"),
                "s.enrollment_max",
                DB::raw("(COUNT(CASE WHEN e.status = 'A' THEN e.id END) * 100.0 / s.enrollment_max) as enrollment_percentage")
            )
            ->groupBy("s.id", "s.enrollment_max", "d.id", "p.id", "t.id");

        if ($yearId > 0) {
            $sql->where("s.year_id", "=", $yearId);
        }
        if ($departmentId > 0) {
            $sql->where("d.id", "=", $departmentId);
        }
        if ($providerId > 0) {
            $sql->where("p.id", "=", $providerId);
        }
        if ($teacherId > 0) {
            $sql->where("s.teacher_id", "=", $teacherId);
        }
        if ($term != 'A') {
            $sql->where("s.term", "=", $term);
        }
        if ($dualCredit != 'A') {
            $sql->where("s.dual_credit", "=", $dualCredit);
        }
        if ($availability != 'A') {
            if ($availability == 'W') {
                $sql->havingRaw('s.enrollment_max > 0 AND enrollment_percentage >= 100');
            } else {
                $sql->havingRaw('s.enrollment_max = 0 OR enrollment_percentage < 100');
            }
        }
        if ($schoolId > 0) {
            $sql->where(function ($query) use ($schoolId) {
                $query->whereNull('s.restricted_school_ids')
                      ->orWhereJsonContains('s.restricted_school_ids', (int) $schoolId);
            });
        }

        switch (auth()->user()->role_id) {
            case 3:
            case 4:
                $sql->where("s.is_cancelled", false);
                break;
            default:
                // Super User can see all enrollments
                // Providers can see all enrollments for their provider
        }

        return $sql->get();
    }

    /**
     * Check if a school is allowed to enroll in this section.
     *
     * @param int $schoolId
     * @return bool
     */
    public function isSchoolAllowed($schoolId): bool
    {
        if (empty($this->restricted_school_ids)) {
            return true;
        }
        return in_array((int) $schoolId, $this->restricted_school_ids, true);
    }

    /**
     * Check if a student is eligible to enroll based on their school.
     *
     * @param Student $student
     * @return bool
     */
    public function canStudentEnroll(Student $student): bool
    {
        return $this->isSchoolAllowed($student->school_id);
    }

    /**
     * Get a formatted string of restricted school names.
     *
     * @return string
     */
    public function getRestrictedSchoolNames(): string
    {
        if (empty($this->restricted_school_ids)) {
            return 'All Schools';
        }

        $schoolNames = [];
        if (in_array(5, $this->restricted_school_ids)) {
            $schoolNames[] = 'BHS';
        }
        if (in_array(6, $this->restricted_school_ids)) {
            $schoolNames[] = 'CHS';
        }
        if (in_array(8, $this->restricted_school_ids)) {
            $schoolNames[] = 'LHS';
        }

        return implode(', ', $schoolNames);
    }
}