<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\User;
use App\Models\Student;
use App\Models\Section;
use App\Models\School;
use App\Models\Year;
use App\Models\Course;
use App\Models\Teacher;
use App\Models\Provider;
use Illuminate\Foundation\Testing\RefreshDatabase;

class EnrollmentRestrictionTest extends TestCase
{
    use RefreshDatabase;

    protected $admin;
    protected $provider;
    protected $schoolUser;
    protected $bhsSchool;
    protected $chsSchool;
    protected $year;
    protected $course;
    protected $teacher;

    protected function setUp(): void
    {
        parent::setUp();

        // Create schools
        $this->bhsSchool = School::factory()->create(['id' => 5, 'name' => 'BHS']);
        $this->chsSchool = School::factory()->create(['id' => 6, 'name' => 'CHS']);

        // Create users
        $this->admin = User::factory()->create(['role_id' => 1]);
        $this->provider = User::factory()->create(['role_id' => 2]);
        $this->schoolUser = User::factory()->create(['role_id' => 3, 'school_id' => $this->bhsSchool->id]);

        // Create supporting models
        $this->year = Year::factory()->create(['id' => 3, 'year' => '2025-2026']);
        $this->course = Course::factory()->create();
        $this->teacher = Teacher::factory()->create();
        $this->provider = Provider::factory()->create();
    }

    /** @test */
    public function role_3_only_sees_allowed_sections()
    {
        // Create sections
        $bhsSection = Section::factory()->create([
            'year_id' => $this->year->id,
            'course_id' => $this->course->id,
            'teacher_id' => $this->teacher->id,
            'provider_id' => $this->provider->id,
            'restricted_school_ids' => [5], // BHS only
            'is_cancelled' => false,
            'term' => 'S',
        ]);
        $chsSection = Section::factory()->create([
            'year_id' => $this->year->id,
            'course_id' => $this->course->id,
            'teacher_id' => $this->teacher->id,
            'provider_id' => $this->provider->id,
            'restricted_school_ids' => [6], // CHS only
            'is_cancelled' => false,
            'term' => 'S',
        ]);

        // Act as role 3 (BHS)
        $response = $this->actingAs($this->schoolUser)->get(route('enrollments.create2', ['yearid' => $this->year->id, 'term' => 'S']));

        // Assert BHS section visible, CHS section not
        $response->assertStatus(200);
        $response->assertSee($bhsSection->course->name);
        $response->assertDontSee($chsSection->course->name);
    }

    /** @test */
    public function role_1_sees_restriction_labels()
    {
        // Create a restricted section
        $section = Section::factory()->create([
            'year_id' => $this->year->id,
            'course_id' => $this->course->id,
            'teacher_id' => $this->teacher->id,
            'provider_id' => $this->provider->id,
            'restricted_school_ids' => [5, 6], // BHS/CHS
            'is_cancelled' => false,
            'term' => 'S',
        ]);

        // Act as role 1 (admin)
        $response = $this->actingAs($this->admin)->get(route('enrollments.create2', ['yearid' => $this->year->id, 'term' => 'S']));

        // Assert restriction label visible
        $response->assertStatus(200);
        $response->assertSee(htmlspecialchars($section->provider->name . ' - ' . $section->course->name . ' - ' . $section->teacher->last_name . ' - Spring - ' . $section->openSeatsDesc() . ' (BHS/CHS ONLY)'));
    }

    /** @test */
    public function role_1_cannot_enroll_ineligible_student()
    {
        // Create a restricted section and students
        $section = Section::factory()->create([
            'year_id' => $this->year->id,
            'course_id' => $this->course->id,
            'teacher_id' => $this->teacher->id,
            'provider_id' => $this->provider->id,
            'restricted_school_ids' => [5], // BHS only
            'is_cancelled' => false,
            'term' => 'S',
            'minimum_grade' => 9,
            'enrollment_max' => 25,
        ]);
        $bhsStudent = Student::factory()->create(['school_id' => 5, 'grade' => 10]);
        $chsStudent = Student::factory()->create(['school_id' => 6, 'grade' => 10]);

        // Act as role 1, try to enroll CHS student
        $response = $this->actingAs($this->admin)->post(route('enrollments.store'), [
            'year_id' => $this->year->id,
            'student_id' => $chsStudent->id,
            'section_id' => $section->id,
            'dual_credit' => false,
            'status' => 'A',
        ]);

        // Assert error message
        $response->assertSessionHasErrors(['student_id' => 'That section is for BHS students only.']);
    }

    /** @test */
    public function role_1_can_enroll_eligible_student()
    {
        // Create a restricted section and student
        $section = Section::factory()->create([
            'year_id' => $this->year->id,
            'course_id' => $this->course->id,
            'teacher_id' => $this->teacher->id,
            'provider_id' => $this->provider->id,
            'restricted_school_ids' => [5], // BHS only
            'is_cancelled' => false,
            'term' => 'S',
            'minimum_grade' => 9,
            'enrollment_max' => 25,
        ]);
        $bhsStudent = Student::factory()->create(['school_id' => 5, 'grade' => 10]);

        // Act as role 1, enroll BHS student
        $response = $this->actingAs($this->admin)->post(route('enrollments.store'), [
            'year_id' => $this->year->id,
            'student_id' => $bhsStudent->id,
            'section_id' => $section->id,
            'dual_credit' => false,
            'status' => 'A',
        ]);

        // Assert successful enrollment
        $response->assertRedirect(route('enrollments.index'));
        $response->assertSessionHas('success', 'Enrollment created successfully!');
        $this->assertDatabaseHas('enrollments', [
            'student_id' => $bhsStudent->id,
            'section_id' => $section->id,
            'status' => 'A',
        ]);
    }
}
